#include <iostream>
#include <cassert>
#include <vector>
#include <algorithm>

using namespace std;

typedef long long ll;

struct Vec {
  ll x, y;
};

struct Line {
  Vec perp;
  ll c;
};

const Vec ZERO = Vec {.x = 0, .y = 0};

Vec operator+ (Vec p, Vec q) {
  return Vec {.x = p.x + q.x, .y = p.y + q.y};
}

Vec operator- (Vec p, Vec q) {
  return Vec {.x = p.x - q.x, .y = p.y - q.y};
}

Vec operator* (int k, Vec p) {
  return Vec {.x = k * p.x, .y = k * p.y};
}

ll cross (Vec p, Vec q) {
  return p.x * q.y - p.y * q.x;
}

ll dot (Vec p, Vec q) {
  return p.x * q.x + p.y * q.y;
}

bool operator== (Vec p, Vec q) {
  return p.x == q.x && p.y == q.y;
}

Vec perp (Vec p) {
  return Vec {.x = -p.y, .y = p.x};
}

int quadrant (Vec p) {
  if (p == ZERO)
    return 0;

  if (p.x > 0 && p.y >= 0)
    return 1;

  if (p.x <= 0 && p.y > 0)
    return 2;

  if (p.x < 0 && p.y <= 0)
    return 3;

  if (p.x >= 0 && p.y < 0)
    return 4;

  assert(false);
}

bool cmp_radial (Vec p, Vec q) {
  if (quadrant(p) != quadrant(q))
    return quadrant(p) < quadrant(q);

  return cross(p, q) > 0;
}

const int N = 7;

int main () {
  ios::sync_with_stdio(false);
  cin.tie(0);

  vector<Vec> pts (N);
  for (int i = 0; i < N; i++)
    cin >> pts[i].x >> pts[i].y;

  vector<Vec> diffs; // O(N^2)
  for (int i = 0; i < N; i++) {
    for (int j = 0; j < N; j++) {
      if (i == j)
        continue;

      Vec diff = pts[i] - pts[j];
      if (diff == ZERO) {
        cout << "EI" << '\n';
        return 0;
      }

      diffs.push_back(diff);
    }
  }

  sort(diffs.begin(), diffs.end(), cmp_radial);

  vector<Vec> dirs; // O(N^2)
  for (int i = 0; i < (int) diffs.size(); i++) {
    Vec a = diffs[i];
    Vec b = diffs[(i + 1) % diffs.size()];

    if (cross(a, b) == 0) {
      if (dot(a, b) < 0) {
        // opposite direction
        dirs.push_back(perp(a));
      }
      // otherwise, same direction; ignore
    } else {
      // points to "between"
      dirs.push_back(a + b);
    }
  }

  vector<Line> lines; // O(N^3)
  for (auto dir : dirs) {
    // find all fundamentally different lines parallel to dir
    vector<ll> cs;
    for (int i = 0; i < N; i++)
      cs.push_back(dot(perp(dir), pts[i]));

    sort(cs.begin(), cs.end());
    for (int i = 0; i < (int) cs.size() - 1; i++) {
      if (cs[i] == cs[i + 1])
        continue;

      // line: perp(dir) * v = (cs[i], cs[i + 1]) / 2;
      lines.push_back(Line {.perp = 2 * perp(dir), .c = -(cs[i] + cs[i + 1]) });
    }
  }

  // O(N^10)
  for (int i = 0; i < (int) lines.size(); i++) {
    for (int j = i + 1; j < (int) lines.size(); j++) {
      for (int k = j + 1; k < (int) lines.size(); k++) {
        // find out if lines[i], lines[j], lines[k] separate the points

        vector<int> cnt (8);
        for (int p = 0; p < N; p++) {
          int mask = 0;
          mask |= dot(lines[i].perp, pts[p]) + lines[i].c > 0;
          mask |= (dot(lines[j].perp, pts[p]) + lines[j].c > 0) << 1;
          mask |= (dot(lines[k].perp, pts[p]) + lines[k].c > 0) << 2;
          cnt[mask]++;
        }

        bool good = true;
        for (int p = 0; p < 8; p++)
          if (cnt[p] > 1)
            good = false;

        if (good) {
          cout << "JAH" << '\n';
          cout << lines[i].perp.x << " " << lines[i].perp.y << " " << lines[i].c << '\n';
          cout << lines[j].perp.x << " " << lines[j].perp.y << " " << lines[j].c << '\n';
          cout << lines[k].perp.x << " " << lines[k].perp.y << " " << lines[k].c << '\n';
          return 0;
        }
      }
    }
  }

  cout << "EI" << '\n';
}
