#include <cstdint>
#include "testlib.h"
#include <numeric>
#include <cassert>

using namespace std;

// generate a dense graph, dropping edges with probability pdrop

// after every change, calculate the set of pairs with changed distances
// and prioritize these when asking queries

typedef long long ll;

const ll INF = 1e18;

struct Edge {
  int u, v, w;
};

const char REMOVE = '-';
const char ADD = '+';
const char ASK = '?';

struct Query {
  char type;
  int u, v;
};

vector<vector<ll>> fw (vector<vector<ll>> dist, vector<int> alive) {
  int n = dist.size();

  for (int k = 0; k < n; k++) {
    if (!alive[k])
      continue;

    for (int i = 0; i < n; i++)
      for (int j = 0; j < n; j++)
        dist[i][j] = min(dist[i][j], dist[i][k] + dist[k][j]);
  }

  return dist;
}

vector<pair<int, int>> diff (const vector<vector<ll>> &A, const vector<vector<ll>> &B) {
  vector<pair<int, int>> ret;
  for (int i = 0; i < (int) min(A.size(), B.size()); i++)
    for (int j = 0; j < (int) min(A[i].size(), B[i].size()); j++)
      if (A[i][j] != B[i][j])
        ret.emplace_back(i, j);

  return ret;
}

vector<pair<int, int>> negatives (const vector<vector<ll>> &A) {
  vector<pair<int, int>> ret;
  for (int i = 0; i < (int) A.size(); i++)
    for (int j = 0; j < (int) A.size(); j++)
      if (A[i][j] == INF)
        ret.emplace_back(i, j);

  return ret;
}

vector<Edge> generate_dense_graph (int n) {
  vector<Edge> edges;
  int wnx = opt<int>("wnx");
  double pdrop = opt<double>("pdrop");
  int lcap = opt<int>("lcap");
  for (int i = 0; i < n; i++)
    for (int j = i + 1; j < n; j++)
      if (rnd.next(0.0, 1.0) > pdrop)
        edges.push_back(Edge {.u = i, .v = j, .w = rnd.wnext(1, lcap, wnx)});
  
  return edges;
}

vector<Edge> generate_tree (int n) {
  vector<Edge> edges;
  int ewnx = opt<int>("ewnx");
  int wwnx = opt<int>("wwnx");
  int lcap = opt<int>("lcap");
  for (int i = 1; i < n; i++) {
    int j = rnd.wnext(i, ewnx);
    edges.push_back(Edge {.u = i, .v = j, .w = rnd.wnext(1, lcap, wwnx)});
  }

  vector<int> perm (n);
  iota(perm.begin(), perm.end(), 0);
  shuffle(perm.begin(), perm.end());

  for (auto &e : edges) {
    e.u = perm[e.u];
    e.v = perm[e.v];
  }

  return edges;
}

vector<int> generate_default_query_template (int n, int q) {
  vector<int> q_template;
  for (int i = 0; i < n; i++) {
    double p = rnd.next(0.0, 1.0);
    int cnt;
    if (p < 0.1)
      cnt = 0;
    else if (p < 0.2)
      cnt = 1;
    else
      cnt = 2;

    for (int k = 0; k < cnt; k++)
      q_template.push_back(i);
  }

  while (q_template.size() < q)
    q_template.push_back(-1);
  
  shuffle(q_template.begin(), q_template.end());
  return q_template;
}

vector<int> generate_valleylike_query_template (int n, int q) {
  int swaps = opt<int>("swaps");
  
  vector<int> enters;
  vector<int> exits;
  for (int i = 0; i < n; i++) {
    double p = rnd.next(0.0, 1.0);
    if (p < 0.05) {
      // do nothing
    } else if (p < 0.2) {
      enters.push_back(i);
    } else {
      enters.push_back(i);
      exits.push_back(i);
    }
  }

  shuffle(enters.begin(), enters.end());
  shuffle(exits.begin(), exits.end());

  vector<int> changes (enters.begin(), enters.end());
  changes.insert(changes.end(), exits.begin(), exits.end());

  // only a small number of swaps to mix it up
  // there will be a period with most vertices being out
  for (int k = 0; k < swaps; k++) {
    int i = rnd.next(changes.size());
    int j = rnd.next(changes.size());
    swap(changes[i], changes[j]);
  }

  vector<int> q_template (changes.size(), -2);
  while ((int) q_template.size() < q)
    q_template.push_back(-1);

  shuffle(q_template.begin(), q_template.end());

  int cptr = 0;
  for (int i = 0; i < (int) q_template.size(); i++) {
    if (q_template[i] == -2) {
      q_template[i] = changes[cptr];
      cptr++;
    }
  }

  return q_template;
}

vector<int> generate_noremove_query_template (int q) {
  return vector<int> (q, -1);
}

vector<int> generate_prefremove_query_template (int n, int q) {
  vector<int> q_template;
  for (int i = 0; i < n; i++) {
    q_template.push_back(i);
  }

  for (int i = 0; i < n; i++) {
    if (rnd.next(0.0, 1.0) > 0.2)
      q_template.push_back(i);
  }

  while (q_template.size() < q)
    q_template.push_back(-1);

  shuffle(q_template.begin(), q_template.begin() + n);
  shuffle(q_template.begin() + n, q_template.end());
  return q_template;
}

int main (int argc, char **argv) {
  registerGen(argc, argv, 1);

  int n = opt<int>("n");
  int q = opt<int>("q");
  bool verbose = __testlib_opts.count("v");

  // graph
  string graph_mode = opt<string>("gmode");
  vector<Edge> edges;
  if (graph_mode == "dense") {
    edges = generate_dense_graph(n);
  } else if (graph_mode == "tree") {
    edges = generate_tree(n);
  } else {
    assert(false);
  }

  vector<vector<ll>> conn (n, vector<ll> (n, INF));
  for (int i = 0; i < n; i++)
    conn[i][i] = 0;

  for (auto e : edges) {
    conn[e.u][e.v] = e.w;
    conn[e.v][e.u] = e.w;
  }

  // queries
  string query_mode = opt<string>("qmode");
  vector<int> q_template;
  if (query_mode == "default") {
    q_template = generate_default_query_template(n, q);
  } else if (query_mode == "noremove") {
    q_template = generate_noremove_query_template(q);
  } else if (query_mode == "prefremove") {
    q_template = generate_prefremove_query_template(n, q);
  } else if (query_mode == "valleylike") {
    q_template = generate_valleylike_query_template(n, q);
  } else {
    assert(false);
  }
  
  vector<int> alive (n, 1);
  auto dist = fw(conn, alive);

  bool skip_smart = __testlib_opts.count("skipsmart");
  
  vector<Query> qs;
  vector<pair<int, int>> pool, negs;
  for (int u : q_template) {
    if (u == -1) {
      double p = rnd.next(0.0, 1.0);
      if (p < 0.2 && !negs.empty()) {
        auto pr = negs[rnd.next(negs.size())];
        qs.push_back(Query {.type = '?', .u = pr.first, .v = pr.second});
      } else if (p < 0.5 && !pool.empty()) {
        auto pr = pool[rnd.next(pool.size())];
        qs.push_back(Query {.type = '?', .u = pr.first, .v = pr.second});
      } else {
        int a, b;
        do {
          a = rnd.next(n);
          b = rnd.next(n);
        } while (a == b);
        qs.push_back(Query {.type = '?', .u = a, .v = b});
      }
    } else {
      qs.push_back(Query {.type = alive[u] ? '-' : '+', .u = u});
      alive[u] ^= 1;
      auto ndist = fw(conn, alive);
      if (!skip_smart) {
        pool = diff(dist, ndist);
        negs = negatives(dist);
        if (verbose)
          cerr << "pool size: " << pool.size() << "; negatives: " << negs.size() << '\n';
      }
      dist = ndist;
    }
  }

  shuffle(edges.begin(), edges.end());
  for (auto &e : edges) {
    if (rnd.next(2)) {
      swap(e.u, e.v);
    }
  }

  cout << n << " " << (int) edges.size() << " " << q << '\n';
  for (auto e : edges) {
    cout << 1 + e.u << " " << 1 + e.v << " " << e.w << '\n';
  }
  for (auto qq : qs) {
    cout << qq.type << " " << 1 + qq.u;
    if (qq.type == '?')
      cout << " " << 1 + qq.v;
    cout << '\n';
  }
}
