#include <iostream>
#include <vector>
#include <algorithm>
#include <cstring>

using namespace std;

// Define limits based on the problem constraints
const int MAX_VAL = 160005;

// Use __int128 for precision in fraction comparison (Standard in CP)
// This avoids floating point errors and overflow issues with large r^k
typedef __int128_t int128;

struct Fraction {
    long long n, d; // numerator, denominator

    // Operator < to compare two fractions: n1/d1 < n2/d2 <=> n1*d2 < n2*d1
    bool operator<(const Fraction &other) const {
        return (int128)n * other.d < (int128)other.n * d;
    }
};

// Global arrays to answer queries in O(1)
Fraction best_prob[MAX_VAL];
int best_id[MAX_VAL];

// DP arrays for the current weapon (Rolling array optimization)
long long dp[MAX_VAL];
long long new_dp[MAX_VAL];

int main() {
    // Fast I/O
    ios_base::sync_with_stdio(false);
    cin.tie(NULL);

    int n, q;
    if (!(cin >> n >> q)) {
        return 0;
    }

    // Initialize best array with impossible probability (-1)
    for (int i = 0; i < MAX_VAL; i++) {
        best_prob[i] = {-1, 1};
        best_id[i] = 1; // Default ID if all probs are 0
    }

    // Process each weapon once
    for (int i = 0; i < n; i++) {
        int k, r, f;
        cin >> k >> r >> f;

        int max_sum = k * r;

        // --- STEP 1: Optimized DP to find ways to get specific sums ---

        // Reset DP for current weapon
        // We assume sum can go up to MAX_VAL. In practice bounded by k*r.
        int limit = min(max_sum, MAX_VAL - 1 - f); // Safety bound

        // Clear DP buffer (only up to previous usage if you want strict optim,
        // but memset is fast enough) We only need to clear up to the max range
        // we used previously, but for simplicity we assume max_sum isn't
        // massive every time. A safe upper bound for clearing is 120000 or
        // track prev_max.
        for (int j = 0; j <= max_sum + 5; ++j) {
            dp[j] = 0;
        }

        dp[0] = 1;
        int cur_max = 0;

        // Iterate through each die
        for (int d = 1; d <= k; d++) {
            long long current_window_sum = 0;
            int prev_max = cur_max;
            cur_max += r;
            if (cur_max >= MAX_VAL) {
                cur_max = MAX_VAL - 1;
            }

            // Sliding window optimization: O(1) transition
            for (int s = 1; s <= cur_max; s++) {
                // Add the new valid index to window
                if (s - 1 <= prev_max) {
                    current_window_sum += dp[s - 1];
                }

                // Remove the index that fell out of window (s - r - 1)
                if (s - r - 1 >= 0) {
                    current_window_sum -= dp[s - r - 1];
                }

                new_dp[s] = current_window_sum;
            }

            // Move new_dp to dp
            for (int s = 0; s <= cur_max; ++s) {
                dp[s] = new_dp[s];
            }
            dp[0] = 0; // sum 0 is impossible for d >= 1
        }

        // --- STEP 2: Calculate Suffix Sums and Update Global Best ---

        // Total combinations = r^k.
        // Instead of pow(), we sum the DP array to be safe against
        // overflow/precision.
        long long total_ways = 0;
        for (int s = k; s <= cur_max; ++s) {
            total_ways += dp[s];
        }

        long long suffix_sum = 0;

        // We iterate backwards from max damage to min damage (k)
        for (int s = cur_max; s >= 0; s--) {
            suffix_sum += dp[s];

            // Map Dice Sum 's' to Monster Health 'H'
            // To kill monster H, we need: dice_sum + f >= H  =>  dice_sum >= H
            // - f So for a required dice sum 's', this covers H = s + f.

            int health = s + f;

            // Boundary checks
            if (health >= MAX_VAL) {
                continue;
            }
            if (health < 0) {
                continue;
            }

            Fraction current_prob = {suffix_sum, total_ways};

            // Update global array if this weapon is better for this specific
            // Health
            if (best_prob[health] < current_prob) {
                best_prob[health] = current_prob;
                best_id[health] = i + 1; // 1-based index
            }
        }

        // --- STEP 3: Handle the "Guaranteed Kill" Range ---
        // If Health <= f + k (min damage), the probability is 100%.
        // The loop above stops at s=0 (health=f). We need to fill 1..f+k
        // explicitly or ensure the loop logic covered it (mostly it covers
        // s>=k). Since s goes down to 0, it covers H down to f. We specifically
        // check the "saturated" range where prob is 1.

        Fraction p1 = {1, 1};
        // The range where prob is 1 is H <= f + k
        int guarantee_limit = min((long long)MAX_VAL - 1, (long long)f + k);

        for (int h = 1; h <= guarantee_limit; ++h) {
            if (best_prob[h] < p1) {
                best_prob[h] = p1;
                best_id[h] = i + 1;
            }
        }
    }

    // --- STEP 4: Answer Queries in O(1) ---
    for (int i = 0; i < q; i++) {
        int t;
        cin >> t;

        if (t >= MAX_VAL) {
            // If H is beyond our array, it's likely impossible for any weapon
            // (assuming MAX_VAL covers max possible damage).
            // Output 1 or whatever fallback is required.
            cout << 1 << "\n";
        } else {
            // Output the precalculated best weapon
            cout << best_id[t] << "\n";
        }
    }

    return 0;
}